/**
  Copyright (c) 2010 Freescale Semiconductor
  
  \file  	  Graphics.c
  \brief	  This is the Graphcis Driver File
  \brief	  Provides functionality to mirror, rotate, sub-scale and copy images
  \author	  Freescale Semiconductor
  \author	  Automotive Systems Solutions Engineering
  \author	  IM, b06623
  \version	  2.0
  \revision	  $Revision: 245 $
  \date  	  $Date: 2011-09-22 17:44:03 -0500 (Thu, 22 Sep 2011) $  
  
  * History:  20/Aug/2008	- Initial Version
	      	  26/Sep/2008	- De-machined, state machines only were adding overhead
	      	  9/March/2009	- Converting functions to non-blocking
	      	  10/March/2009 - Adding combined rotation with mirror operations
	      	  13/May/2009 	- MISRA Check. Release
	      	  25/May/2010 	- Update to improve performance. Removed usage of cfg structures and
              				  functions from the DMA
              April-2010  	- Support DCU3 bpp modes (IM)
              Ago-2010    	- Support DCU3 tile, rle, partial blank area functionality (IM) 
              Nov-2010		- Added support for YCBCR422 & APAL8
             				  

  * MISRA VIOLATIONS:
	- [ MISRA 16.9 ]

* Copyright (c) 2010, Freescale, Inc.  All rights reserved.
*
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
*
   
*/

#include	"Graphics.h"


uint32_t Graphics_BlankValue32[8];
uint8_t Graphics_BlankValueIdx;

Graphics_CallbackType Graphics_TCB[DMA_MAX_DMACHANNELS ];

static void Graphics_Pixel2Bytes(uint8_t BPP, uint32_t *size);
static void Graphics_TransformationCallback(uint8_t channel);

/**
* \brief	Graphics_Init - Initializes the graphics library control variables
* \brief	IT HAS NO FUNCTIONALITY YET
* \author	IM, b06623
* \param	void
* \return	void
* \todo
*/
void Graphics_Init(void)
{
    
}

/**
* \brief	Display_CopyGraphicObject - This function copies all the data from one G-object to another.
* \author	IM, b06623
* \param	const Graphics_Object_t* source: Is the source object
* \param        Graphics_Object_t* target: Is the target object
* \return	void
* \todo
*/
void Graphics_CopyGraphicObject(const Graphics_Object_t *source, Graphics_Object_t *target)
{
    target->address = source->address;
    target->width = source->width;
    target->height = source->height;
    target->CLUT = source->CLUT;
    target->CLUTsize = source->CLUTsize;
    target->CLUTOffset = source->CLUTOffset;
    target->BPP = source->BPP;
    target->initialAlpha = source->initialAlpha;
    target->initialChromaMax = source->initialChromaMax;
    target->initialChromaMin = source->initialChromaMin;
    target->coding = source->coding;
    target->x = source->x;
    target->y = source->y;  
    target->ext = source->ext;    
}

/**
* \brief	Graphics_BlankArea - Blanks a memory area with the selected value.
* \brief	To correctly fill 32bpp areas, destination must be aligned to 32bits
* \author	IM, b06623
* \param	uint8_t Channel, it is the DMA channel to be initialized.
* \param	const Graphics_Object_t* g_object: Is the graphic object info to initialize the layer
* \param	uint32_t value: This is the filling value, this pattern will be used to fill the memory
* \return	DMA_ErrorType, possible resutls: GRAPHICS_ERROR_OK, GRAPHICS_ERROR_FAIL
* \todo
*/
Graphics_ErrorType Graphics_BlankArea
(
	uint8_t eDMAChannel, 
	const Graphics_Object_t*  g_object, 
	uint32_t value, 
	Graphics_CallbackType callback
)
{
	//DMA_ChannelCfgType cfg1;
	DMA_ErrorType	   error;
	Graphics_ErrorType error1;
	uint8_t * dest;
	uint32_t size;

	error1 = GRAPHICS_ERROR_OK;

	error  = DMA_InitChannel( eDMAChannel );
	if( error != DMA_ERROR_OK )
	{
		error1 = GRAPHICS_ERROR_FAIL;
	}
	else
	{
		Graphics_BlankValueIdx++;
		Graphics_BlankValue32[Graphics_BlankValueIdx&7] = value;
		dest = (uint8_t *)g_object->address;  

		DMA_DestinationAddress(eDMAChannel) = ((g_object->address/4)*4)+4;
		DMA_SourceAddress(eDMAChannel) 		= ( uint32_t )&(Graphics_BlankValue32[Graphics_BlankValueIdx&7]);
			
		/* 32 bits transfer */
		DMA_SourceSize( eDMAChannel ) 	   	   = 2u;
		DMA_DestinationSize( eDMAChannel ) 	   = 2u; 

		size = (uint32_t)g_object->height * (uint32_t)g_object->width;
		Graphics_Pixel2Bytes(g_object->BPP, &size);
		
		DMA_DestinationOffset(eDMAChannel) 	= 4;
		DMA_SourceOffset(eDMAChannel) 		= 0;
		DMA_SetMajorLoopCount(eDMAChannel,1);
		DMA_MinorLoopCount(eDMAChannel)		= ((size>>2)<<2)-4;;			

		/* Modify non-aligned begining and end data */
		dest[0] = (uint8_t)(value>>24u);  
		dest[1] = (uint8_t)(value>>16u); 
		dest[2] = (uint8_t)(value>>8u); 
		dest[3] = (uint8_t)(value); 

		dest[size - 4u] = (uint8_t)(value>>24u);  
		dest[size - 3u] = (uint8_t)(value>>16u); 
		dest[size - 2u] = (uint8_t)(value>>8u); 
		dest[size - 1u] = (uint8_t)(value);
		
		
		if(DMA_MinorLoopCount(eDMAChannel) == 0u)
		{
			error1 = GRAPHICS_ERROR_FAIL;
		}
		else
		{
        		/* VIOLATION TO THE FOLLOWING RULE (16.9-1)*/ 
        		/* [ MISRA 16.9 ] A function identifier shall only be used with either a preceding &, */
				/* or with a parenthesised parameter list, which may be empty. */
                /* Rationale: There is no work-around to configure a run-time callback */ 
				Graphics_TCB[eDMAChannel] =  callback; 
				DMA_SetCallback(eDMAChannel, Graphics_TransformationCallback);
		        DMA_SetReady(eDMAChannel);
		}	
	}
	return error1;
}

/**
* \brief	Graphics_BlankArea_Rect - Blanks a memory area with the selected value, with the selected coordinates.
* \brief	To correctly fill 32bpp areas, destination must be aligned to 32bits
* \author	IM, b06623
* \param	uint8_t Channel, it is the DMA channel to be initialized.
* \param	const Graphics_Object_t* g_object: Is the graphic object info to initialize the layer
* \param	uint16_t xi: It is the first x coordinate.
* \param	uint16_t yi: It is the first y coordinate.
* \param	uint16_t xf: It is the final x coordinate.
* \param	uint16_t yf: It is the final y coordinate.
* \param	uint32_t value: This is the filling value, this pattern will be used to fill the memory
* \return	DMA_ErrorType, possible resutls: GRAPHICS_ERROR_OK, GRAPHICS_ERROR_FAIL
* \todo
*/
uint32_t Graphics_BA_Rect;
Graphics_ErrorType Graphics_BlankArea_Rect
(
	uint8_t eDMAChannel, 
	const Graphics_Object_t*  g_object,
	uint16_t xi,
	uint16_t yi,
	uint16_t xf,
	uint16_t yf, 
	uint32_t value, 
	Graphics_CallbackType callback
)
{
	DMA_ErrorType	   error;
	Graphics_ErrorType error1;
	uint32_t size;
	

	error1 = GRAPHICS_ERROR_OK;
	size = 8;
	Graphics_Pixel2Bytes(g_object->BPP, &size);

	error  = DMA_InitChannel( eDMAChannel );
	if( error != DMA_ERROR_OK )
	{
		error1 = GRAPHICS_ERROR_FAIL;
	}
    /* Coordinates Validation */
    else if( ( (xi >= xf) || (yi >= yf) ) || ((xf - xi) > g_object->width) )
    {
		DMA_UnInitChannel(eDMAChannel);
		error1 = GRAPHICS_ERROR_FAIL;
    }
    else if(size < 4)
    {
		DMA_UnInitChannel(eDMAChannel);
		error1 = GRAPHICS_ERROR_FAIL;
    } 	
	else
	{
		
		Graphics_BA_Rect = value;
		DMA_DestinationAddress(eDMAChannel)	= (((xi + (yi*( uint32_t )g_object->width))*(int32_t)size)>>3) + g_object->address;
		DMA_SourceAddress(eDMAChannel) 		= (uint32_t)(&Graphics_BA_Rect);
			
		/* 8 bits transfer */
		DMA_SourceMod( eDMAChannel ) 		   = 2;
		DMA_SourceSize( eDMAChannel ) 	   	   = 0;
		DMA_DestinationSize( eDMAChannel ) 	   = 0;
		
		DMA_MinorLoopCount(eDMAChannel)		= (((uint32_t)xf - (uint32_t)xi)*size)>>3;
		DMA_SetMajorLoopCount(eDMAChannel,((uint16_t)(yf - yi)));
		
		DMA_MdestOffsetBit(eDMAChannel) = 1;
		DMA_MsrcOffsetBit(eDMAChannel) = 0; 
		DMA_MloopOffsetValue(eDMAChannel) = (((int32_t)g_object->width - ((int32_t)xf - (int32_t)xi))*(int32_t)size)>>3;		
		
		DMA_DestinationOffset(eDMAChannel) 	= 1;
		DMA_SourceOffset(eDMAChannel) 		= 1;	
		
		Graphics_TCB[eDMAChannel] =  callback; 
		DMA_SetCallback(eDMAChannel, Graphics_TransformationCallback);
		size = (uint32_t)(yf-yi);
		if(size > 1)
		{
			DMA_EnableRequest(eDMAChannel, (uint8_t)(DMA_REQ_ALWAYS_START + eDMAChannel));
		}
		DMA_SetReady(eDMAChannel);		
	}
	
	return error1;
}

/**
* \brief	Graphics_SubCopy - Takes Copies a section (equal or lesser) 
* \brief	from graphic object to another graphic object
* \brief	IMPORTANT: Target with must be the same than delta 'x'
* \author	IM, b06623
* \param	uint8_t Channel, it is the DMA channel to be used.
* \param	const Graphics_Object_t* Source: Is the graphic object Source to be copied
* \param	const Graphics_Object_t* target: Is the graphic object Destination
* \param	uint16_t xi: It is the first x coordinate.
* \param	uint16_t yi: It is the first y coordinate.
* \param	uint16_t xf: It is the final x coordinate.
* \param	uint16_t yf: It is the final y coordinate.
* \return	DMA_ErrorType, possible resutls: GRAPHICS_ERROR_OK, GRAPHICS_ERROR_FAIL
* \todo
*/
Graphics_ErrorType Graphics_SubCopy
(
	uint8_t  eDMAChannel,
	Graphics_Object_t*  Source, 
	Graphics_Object_t*  Target,
	uint16_t xi,
	uint16_t yi,
	uint16_t xf,
	uint16_t yf,
	Graphics_CallbackType callback
)
{
    DMA_ErrorType	   	error;    
    Graphics_ErrorType	error1;
    uint32_t			tmp;

    error1 = GRAPHICS_ERROR_OK;
    error  = DMA_InitChannel( eDMAChannel );
   	tmp = 8u;
	Graphics_Pixel2Bytes(Target->BPP, &tmp);
    
    if( error != DMA_ERROR_OK )
    {
		error1 = GRAPHICS_ERROR_FAIL;
    }
    /* Coordinates Validation */
    else if( ( (xi >= xf) || (yi >= yf) ) || ((xf - xi) != Target->width) )
    {
		DMA_UnInitChannel(eDMAChannel);
		error1 = GRAPHICS_ERROR_FAIL;
    }
    else if(Target->BPP != Source->BPP)
    {
		DMA_UnInitChannel(eDMAChannel);
		error1 = GRAPHICS_ERROR_FAIL;
    }
    else if((tmp < 8u) || (tmp == 24u))
    {
		DMA_UnInitChannel(eDMAChannel);
		error1 = GRAPHICS_ERROR_FAIL;
    }    
    else
    {
    
    	if(tmp == 8u)
		{
		    DMA_DestinationSize(eDMAChannel) = 0u;
		    DMA_SourceSize(eDMAChannel) = 0u;
		    tmp = 1u;
		}
		else if(tmp == 16u)
		{
		    DMA_DestinationSize(eDMAChannel) = 1u;
		    DMA_SourceSize(eDMAChannel) = 1u;
		    tmp = 2u;
		}
		else /* if(tmp == 32) */
		{
			DMA_DestinationSize(eDMAChannel) = 2u;
			DMA_SourceSize(eDMAChannel) = 2u;
			tmp = 4u;
		}

		DMA_DestinationOffset(eDMAChannel) 	= (int16_t)tmp;
		DMA_SourceOffset(eDMAChannel) 		= (int16_t)tmp;
		
		DMA_MinorLoopCount(eDMAChannel)		= ((uint32_t)xf - (uint32_t)xi)*tmp;
		DMA_SetMajorLoopCount(eDMAChannel,((uint16_t)(yf - yi)));
		
		DMA_MdestOffsetBit(eDMAChannel) = 0;
		DMA_MsrcOffsetBit(eDMAChannel) = 1; 
		DMA_MloopOffsetValue(eDMAChannel) = ((int32_t)Source->width - ((int32_t)xf - (int32_t)xi))*(int32_t)tmp;
		
		DMA_DestinationAddress(eDMAChannel) = Target->address;
		DMA_SourceAddress(eDMAChannel)	= ((xi + (yi*( uint32_t )Source->width))*(int32_t)tmp)+ Source->address;
		
		/* VIOLATION TO [ MISRA 16.9 ] See 16.9-1 */
		Graphics_TCB[eDMAChannel] =  callback; 
		DMA_SetCallback(eDMAChannel, Graphics_TransformationCallback);
		DMA_EnableRequest(eDMAChannel, (uint8_t)(DMA_REQ_ALWAYS_START + eDMAChannel));
		DMA_SetReady(eDMAChannel);						
    }
    return error1;    
}

/**
* \brief	Graphics_Paste - Paste a GO where indicated. The source.width must be smaller than the target's
* \author	IM, b06623
* \param	uint8_t Channel, it is the DMA channel to be used.
* \param	const Graphics_Object_t* Source: Is the graphic object Source to be copied
* \param	const Graphics_Object_t* target: Is the graphic object Destination
* \param	uint16_t xi: It is the first x coordinate.
* \param	uint16_t yi: It is the first y coordinate.
* \return	DMA_ErrorType, possible resutls: GRAPHICS_ERROR_OK, GRAPHICS_ERROR_FAIL
* \todo
*/
Graphics_ErrorType Graphics_Paste
(
	uint8_t  eDMAChannel,
	Graphics_Object_t*  Source, 
	Graphics_Object_t*  Target,
	uint16_t xi,
	uint16_t yi,
	Graphics_CallbackType callback
)
{
    //DMA_ChannelCfgType cfg1;
    DMA_ErrorType	   error;
    Graphics_ErrorType error1;
    uint32_t tmp;
    uint32_t tmp0;
    
    error1 = GRAPHICS_ERROR_OK;
    error  = DMA_InitChannel( eDMAChannel );
    /* Calculate pixel size */
    tmp = 8u;
    Graphics_Pixel2Bytes(Target->BPP, &tmp);
    
    if( error != DMA_ERROR_OK )
    {
	error1 = GRAPHICS_ERROR_FAIL;
    }
    else if(Source->width > Target->width)
    {
	DMA_UnInitChannel(eDMAChannel);
	error1 = GRAPHICS_ERROR_FAIL;
    }
    else if((tmp < 4u) || (tmp == 24u))
    {
	DMA_UnInitChannel(eDMAChannel);
	error1 = GRAPHICS_ERROR_FAIL;
    }
    else
    {
	if(tmp == 4u)
	{
	    DMA_DestinationSize(eDMAChannel) = 0u;
	    DMA_SourceSize(eDMAChannel) = 0u;
	    tmp = 1u;
	    tmp0 = 2u;
	}
	else if(tmp == 8u)
	{
	    DMA_DestinationSize(eDMAChannel) = 0u;
	    DMA_SourceSize(eDMAChannel) = 0u;
	    tmp = 1u;
	    tmp0 = 1u;
	}
	else if(tmp == 16u)
	{
	    DMA_DestinationSize(eDMAChannel) = 1u;
	    DMA_SourceSize(eDMAChannel) = 1u;
	    tmp = 2u;
	    tmp0 = 1u;
	}
	else /* if(tmp == 32) */
	{
	   DMA_DestinationSize(eDMAChannel) = 2u;
	   DMA_SourceSize(eDMAChannel) = 2u;
	   tmp = 4u;
	   tmp0 = 1u;
	}


	DMA_DestinationOffset(eDMAChannel) 	= (int16_t)tmp;
	DMA_SourceOffset(eDMAChannel) 		= (int16_t)tmp;
	DMA_SourceAddress(eDMAChannel) 	= Source->address;


	if(tmp0 == 2)
	{
		DMA_MinorLoopCount(eDMAChannel) = (uint32_t)(Source->width/tmp0);
		DMA_MloopOffsetValue(eDMAChannel) = ((int32_t)Target->width/(int32_t)tmp0) - (int32_t)(Source->width/tmp0);		
		DMA_DestinationAddress(eDMAChannel)	= Target->address + ((xi + (yi*Target->width))/tmp0);		
	}
	else
	{
		DMA_MinorLoopCount(eDMAChannel) = (uint32_t)Source->width*tmp;
		DMA_MloopOffsetValue(eDMAChannel) = ((int32_t)Target->width - (int32_t)Source->width)*(int32_t)tmp;		
		DMA_DestinationAddress(eDMAChannel)	= Target->address + ((xi + (yi*Target->width))*tmp);
	}
	
	DMA_MdestOffsetBit(eDMAChannel) = 1;
	DMA_MsrcOffsetBit(eDMAChannel) = 0; 

	if( Target->height < Source->height )
	{
	    DMA_SetMajorLoopCount(eDMAChannel,Target->height);
	}
	else
	{
	    DMA_SetMajorLoopCount(eDMAChannel,Source->height);
	}

	
	/* VIOLATION TO [ MISRA 16.9 ] See 16.9-1*/
	Graphics_TCB[eDMAChannel] =  callback; 
	DMA_SetCallback(eDMAChannel, Graphics_TransformationCallback);
    DMA_EnableRequest(eDMAChannel, (uint8_t)(DMA_REQ_ALWAYS_START + eDMAChannel));
    DMA_SetReady(eDMAChannel);

    }
    return error1;
}

/**
* \brief	Graphics_Rotate90 - Rotates a graphic object 90 degrees ccwise
* \brief	The width of the source cannot be higher than the target height
* \author	IM, b06623
* \param	uint8_t Channel, it is the DMA channel to be used.
* \param	const Graphics_Object_t* Source: Is the graphic object Source to be copied
* \param	const Graphics_Object_t* target: Is the graphic object Destination
* \return	DMA_ErrorType, possible resutls: GRAPHICS_ERROR_OK, GRAPHICS_ERROR_FAIL
* \todo
*/
Graphics_ErrorType Graphics_Rotate90( uint8_t  eDMAChannel, Graphics_Object_t* Source, Graphics_Object_t*  Target, Graphics_CallbackType callback)
{
    DMA_ErrorType	error;
    Graphics_ErrorType 	error1;
    uint32_t		tmp;

    error1 = GRAPHICS_ERROR_OK;
    error  = DMA_InitChannel( eDMAChannel );
    /* Calculate pixel size */
    tmp = 8u;
    Graphics_Pixel2Bytes(Target->BPP, &tmp);

    if( error != DMA_ERROR_OK )
    {
		error1 = GRAPHICS_ERROR_FAIL;
    }
    else if(Source->width > Target->height)
    {
		DMA_UnInitChannel(eDMAChannel);
		error1 = GRAPHICS_ERROR_FAIL;
    }
    else if((tmp < 8u) || (tmp == 24u))
    {
		DMA_UnInitChannel(eDMAChannel);
		error1 = GRAPHICS_ERROR_FAIL;
    }
    else
    {
		if(tmp == 8u)
		{
		    DMA_DestinationSize(eDMAChannel) = 0u;
		    DMA_SourceSize(eDMAChannel) = 0u;
		    tmp = 1u;
		}
		else if(tmp == 16u)
		{
		    DMA_DestinationSize(eDMAChannel) = 1u;
		    DMA_SourceSize(eDMAChannel) = 1u;
		    tmp = 2u;
		}
		else /* if(tmp == 32) */
		{
		   DMA_DestinationSize(eDMAChannel) = 2u;
		   DMA_SourceSize(eDMAChannel) = 2u;
		   tmp = 4u;
		}
		

		DMA_SourceOffset(eDMAChannel) = (int16_t)tmp; 
		DMA_DestinationOffset(eDMAChannel)  = (int16_t)((int16_t)Target->width*(int16_t)tmp); 
		
		
		DMA_MinorLoopCount(eDMAChannel) = (uint32_t)Source->width*tmp;	
		DMA_MloopOffsetValue(eDMAChannel) = -1*((int32_t)tmp + (((int32_t)Target->width*(int32_t)tmp) * (int32_t)Source->width));
		
		DMA_SourceAddress(eDMAChannel) = Source->address; 
		DMA_DestinationAddress(eDMAChannel) = Target->address - tmp + (uint32_t)Target->width*tmp;	

		if( Target->width < Source->height )
		{
		    DMA_SetMajorLoopCount(eDMAChannel,Target->height);
		    
		}
		else
		{
		    DMA_SetMajorLoopCount(eDMAChannel,Source->height);
		}
		
		DMA_MdestOffsetBit(eDMAChannel) = 1;
		DMA_MsrcOffsetBit(eDMAChannel) = 0; 

		/* VIOLATION TO [ MISRA 16.9 ] See 16.9-1*/
		Graphics_TCB[eDMAChannel] =  callback; 
		DMA_SetCallback(eDMAChannel, Graphics_TransformationCallback);
	    DMA_EnableRequest(eDMAChannel, (uint8_t)(DMA_REQ_ALWAYS_START + eDMAChannel));
	    DMA_SetReady(eDMAChannel);

    }
    return error1;
}

/**
* \brief	Graphics_Rotate180 - Rotates a graphic object 90 degrees ccwise
* \author	IM, b06623
* \param	uint8_t Channel, it is the DMA channel to be used.
* \param	const Graphics_Object_t* Source: Is the graphic object Source to be copied
* \param	const Graphics_Object_t* target: Is the graphic object Destination
* \return	DMA_ErrorType, possible resutls: GRAPHICS_ERROR_OK, GRAPHICS_ERROR_FAIL
* \todo
*/
Graphics_ErrorType Graphics_Rotate180( uint8_t	eDMAChannel, Graphics_Object_t* Source, Graphics_Object_t* Target, Graphics_CallbackType callback)
{
    DMA_ErrorType		error;
    Graphics_ErrorType 	error1;
    uint32_t			tmp;

    error1 = GRAPHICS_ERROR_OK;
    error  = DMA_InitChannel( eDMAChannel );
    /* Calculate pixel size */
    tmp = 8u;
    Graphics_Pixel2Bytes(Target->BPP, &tmp);
    
    if( error != DMA_ERROR_OK )
    {
		error1 = GRAPHICS_ERROR_FAIL;
    }
    else if(Source->width > Target->width)
    {
		DMA_UnInitChannel(eDMAChannel);
		error1 = GRAPHICS_ERROR_FAIL;
    }
    else if((tmp < 8u) || (tmp == 24u))
    {
		DMA_UnInitChannel(eDMAChannel);
		error1 = GRAPHICS_ERROR_FAIL;
    }
    else
    {
		if(tmp == 8u)
		{
		    DMA_DestinationSize(eDMAChannel) = 0u;
		    DMA_SourceSize(eDMAChannel) = 0u;
		    tmp = 1u;
		}
		else if(tmp == 16u)
		{
		    DMA_DestinationSize(eDMAChannel) = 1u;
		    DMA_SourceSize(eDMAChannel) = 1u;
		    tmp = 2u;
		}
		else /* if(tmp == 32) */
		{
		    DMA_DestinationSize(eDMAChannel) = 2u;
		    DMA_SourceSize(eDMAChannel) = 2u;
		    tmp = 4u;
		}
		
		
		DMA_SourceOffset(eDMAChannel) = (int16_t)tmp; 
		DMA_DestinationOffset(eDMAChannel)  = (int16_t)(-1*(int16_t)tmp);
		
		
		DMA_MinorLoopCount(eDMAChannel) = (uint32_t)Source->width*tmp;
		DMA_MloopOffsetValue(eDMAChannel) = (((int32_t)Source->width) - ((int32_t)Target->width))*(int32_t)tmp;
		
		DMA_SourceAddress(eDMAChannel) = Source->address; 
		DMA_DestinationAddress(eDMAChannel) = Target->address + ((uint32_t)(Target->height * ((uint32_t)Target->width*tmp))) - tmp;

		if( Target->height < Source->height )
		{
		    DMA_SetMajorLoopCount(eDMAChannel,Target->height);
		}
		else
		{
		    DMA_SetMajorLoopCount(eDMAChannel,Source->height);
		}

		DMA_MdestOffsetBit(eDMAChannel) = 1;
		DMA_MsrcOffsetBit(eDMAChannel) = 0; 

		/* VIOLATION TO [ MISRA 16.9 ] See 16.9-1*/
		Graphics_TCB[eDMAChannel] =  callback; 
		DMA_SetCallback(eDMAChannel, Graphics_TransformationCallback);
	    DMA_EnableRequest(eDMAChannel, (uint8_t)(DMA_REQ_ALWAYS_START+ eDMAChannel));
	    DMA_SetReady(eDMAChannel);		    
    }
    return error1;
}

/**
* \brief	Graphics_Rotate270 - Rotates a graphic object 270 degrees ccwise
* \brief	The width of the source cannot be higher than the target height 
* \author	IM, b06623
* \param	uint8_t Channel, it is the DMA channel to be used.
* \param	const Graphics_Object_t* Source: Is the graphic object Source to be copied
* \param	const Graphics_Object_t* target: Is the graphic object Destination
* \return	DMA_ErrorType, possible resutls: GRAPHICS_ERROR_OK, GRAPHICS_ERROR_FAIL
* \todo
*/
Graphics_ErrorType Graphics_Rotate270(uint8_t eDMAChannel, Graphics_Object_t* Source, Graphics_Object_t* Target, Graphics_CallbackType callback)
{
    DMA_ErrorType		error;
    Graphics_ErrorType 	error1;
    uint32_t 			tmp;
    
    error1 = GRAPHICS_ERROR_OK;
    error  = DMA_InitChannel( eDMAChannel );
    /* Calculate pixel size */
    tmp = 8u;
    Graphics_Pixel2Bytes(Target->BPP, &tmp);   

    if( error != DMA_ERROR_OK )
    {
		error1 = GRAPHICS_ERROR_FAIL;
    }
    else if(Source->width > Target->height)
    {
		DMA_UnInitChannel(eDMAChannel);
		error1 = GRAPHICS_ERROR_FAIL;
    }
    else if((tmp < 8u) || (tmp == 24u))
    {
		DMA_UnInitChannel(eDMAChannel);
		error1 = GRAPHICS_ERROR_FAIL;
    }
    else
    {
	if(tmp == 8u)
	{
	    DMA_DestinationSize(eDMAChannel) = 0u;
	    DMA_SourceSize(eDMAChannel) = 0u;
	    tmp = 1u;
	}
	else if(tmp == 16u)
	{
	    DMA_DestinationSize(eDMAChannel) = 1u;
	    DMA_SourceSize(eDMAChannel) = 1u;
	    tmp = 2u;
	}
	else /* if(tmp == 32) */
	{
	   DMA_DestinationSize(eDMAChannel) = 2u;
	   DMA_SourceSize(eDMAChannel) = 2u;
	   tmp = 4u;
	}
	
	
		DMA_SourceOffset(eDMAChannel) = (int16_t)tmp; 
		DMA_DestinationOffset(eDMAChannel)  = (int16_t)(-1*(int16_t)Target->width*(int16_t)tmp);
		
		
		DMA_MinorLoopCount(eDMAChannel) = (uint32_t)Source->width*tmp;
		DMA_MloopOffsetValue(eDMAChannel) = (int32_t)tmp + (((int32_t)Source->width) * ((int32_t)Target->width*(int32_t)tmp));
		
		DMA_SourceAddress(eDMAChannel) = Source->address; 
		DMA_DestinationAddress(eDMAChannel) = Target->address + (((uint32_t)Source->width - 1u) * ((uint32_t)Target->width*tmp));

		if( Target->width < Source->height )
		{
		    DMA_SetMajorLoopCount(eDMAChannel,Target->width);
		}
		else
		{
		    DMA_SetMajorLoopCount(eDMAChannel,Source->height);
		}

		DMA_MdestOffsetBit(eDMAChannel) = 1;
		DMA_MsrcOffsetBit(eDMAChannel) = 0; 

		/* VIOLATION TO [ MISRA 16.9 ] See 16.9-1*/
		Graphics_TCB[eDMAChannel] =  callback; 
		DMA_SetCallback(eDMAChannel, Graphics_TransformationCallback);
	    DMA_EnableRequest(eDMAChannel, (uint8_t)(DMA_REQ_ALWAYS_START + eDMAChannel));
	    DMA_SetReady(eDMAChannel);	

    }
    return error1;
}

/**
* \brief	Graphics_MirrorV - Mirrors a graphic object vertically (flip vertical)
* \author	IM, b06623
* \param	uint8_t Channel, it is the DMA channel to be used.
* \param	const Graphics_Object_t* Source: Is the graphic object Source to be copied
* \param	const Graphics_Object_t* target: Is the graphic object Destination
* \return	DMA_ErrorType, possible resutls: GRAPHICS_ERROR_OK, GRAPHICS_ERROR_FAIL
* \todo
*/
Graphics_ErrorType Graphics_MirrorV(uint8_t eDMAChannel, Graphics_Object_t * Source, Graphics_Object_t * Target, Graphics_CallbackType callback)
{
    DMA_ErrorType	   error;
    Graphics_ErrorType error1;
    uint32_t tmp;
    
    error1 = GRAPHICS_ERROR_OK;
    error  = DMA_InitChannel( eDMAChannel );
    /* Calculate pixel size */
    tmp = 8u;
    Graphics_Pixel2Bytes(Target->BPP, &tmp);
    
    if( error != DMA_ERROR_OK )
    {
		error1 = GRAPHICS_ERROR_FAIL;
    }
    else if(Source->width > Target->width)
    {
		DMA_UnInitChannel(eDMAChannel);
		error1 = GRAPHICS_ERROR_FAIL;
    }
    else if((tmp < 8u) || (tmp == 24u))
    {
		DMA_UnInitChannel(eDMAChannel);
		error1 = GRAPHICS_ERROR_FAIL;
    }
    else
    {
		if(tmp == 8u)
		{
		    DMA_DestinationSize(eDMAChannel) = 0u;
		    DMA_SourceSize(eDMAChannel) = 0u;
		    tmp = 1u;
		}
		else if(tmp == 16u)
		{
		    DMA_DestinationSize(eDMAChannel) = 1u;
		    DMA_SourceSize(eDMAChannel) = 1u;
		    tmp = 2u;
		}
		else /* if(tmp == 32) */
		{
		   DMA_DestinationSize(eDMAChannel) = 2u;
		   DMA_SourceSize(eDMAChannel) = 2u;
		   tmp = 4u;
		}
				
		DMA_SourceOffset(eDMAChannel) = (int16_t)tmp; 
		DMA_DestinationOffset(eDMAChannel)  = (int16_t)(-1*(int16_t)tmp);
		
		DMA_MinorLoopCount(eDMAChannel) = (uint32_t)Source->width*tmp;
		DMA_MloopOffsetValue(eDMAChannel) = ((int32_t)Target->width + (int32_t)Source->width)*(int32_t)tmp;
		
		DMA_SourceAddress(eDMAChannel) = Source->address; 
		DMA_DestinationAddress(eDMAChannel) = Target->address - tmp + ((uint32_t)Source->width*tmp);
		
		if( Target->height < Source->height )
		{
		    DMA_SetMajorLoopCount(eDMAChannel,Target->height);
		}
		else
		{
		    DMA_SetMajorLoopCount(eDMAChannel,Source->height);
		}

		DMA_MdestOffsetBit(eDMAChannel) = 1;
		DMA_MsrcOffsetBit(eDMAChannel) = 0; 

		/* VIOLATION TO [ MISRA 16.9 ] See 16.9-1*/
		Graphics_TCB[eDMAChannel] =  callback; 
		DMA_SetCallback(eDMAChannel, Graphics_TransformationCallback);
	    DMA_EnableRequest(eDMAChannel, (uint8_t)(DMA_REQ_ALWAYS_START + eDMAChannel));
	    DMA_SetReady(eDMAChannel);	
    }
    return error1;
}


/**
* \brief	Graphics_MirrorH - Mirrors a graphic object horizontally (flip horizontal)
* \author	IM, b06623
* \param	uint8_t Channel, it is the DMA channel to be used.
* \param	const Graphics_Object_t* Source: Is the graphic object Source to be copied
* \param	const Graphics_Object_t* target: Is the graphic object Destination
* \return	DMA_ErrorType, possible resutls: GRAPHICS_ERROR_OK, GRAPHICS_ERROR_FAIL
* \todo
*/
Graphics_ErrorType Graphics_MirrorH( uint8_t  eDMAChannel, Graphics_Object_t*  Source, Graphics_Object_t*  Target, Graphics_CallbackType callback)
{
    DMA_ErrorType	error;
    Graphics_ErrorType 	error1;
    uint32_t tmp;
    
    error1 = GRAPHICS_ERROR_OK;
    error  = DMA_InitChannel( eDMAChannel );
    /* Calculate pixel size */
    tmp = 8u;
    Graphics_Pixel2Bytes(Target->BPP, &tmp);
    
    if( error != DMA_ERROR_OK )
    {
		error1 = GRAPHICS_ERROR_FAIL;
    }
    else if(Source->width > Target->width)
    {
		DMA_UnInitChannel(eDMAChannel);
		error1 = GRAPHICS_ERROR_FAIL;
    }
    else if((tmp < 8u) || (tmp == 24u))
    {
		DMA_UnInitChannel(eDMAChannel);
		error1 = GRAPHICS_ERROR_FAIL;
    }
    else
    {
		if(tmp == 8u)
		{
		    DMA_DestinationSize(eDMAChannel) = 0u;
		    DMA_SourceSize(eDMAChannel) = 0u;
		    tmp = 1u;
		}
		else if(tmp == 16u)
		{
		    DMA_DestinationSize(eDMAChannel) = 1u;
		    DMA_SourceSize(eDMAChannel) = 1u;
		    tmp = 2u;
		}
		else /* if(tmp == 32) */
		{
		   DMA_DestinationSize(eDMAChannel) = 2u;
		   DMA_SourceSize(eDMAChannel) = 2u;
		   tmp = 4u;
		}
	
		DMA_SourceOffset(eDMAChannel) = (int16_t)tmp; 
		DMA_DestinationOffset(eDMAChannel)  = (int16_t)tmp; 
		
		DMA_MinorLoopCount(eDMAChannel) = (uint32_t)Source->width*tmp;
		DMA_MloopOffsetValue(eDMAChannel) = (-1*((int32_t)Target->width + (int32_t)Source->width))*(int32_t)tmp;
		
		DMA_SourceAddress(eDMAChannel) = Source->address; 
		DMA_DestinationAddress(eDMAChannel) = Target->address + (((uint32_t)Target->width*tmp) * ( uint32_t )(Target->height - 1u));
		
		if( Target->height < Source->height )
		{
		    DMA_SetMajorLoopCount(eDMAChannel,Target->height);
		}
		else
		{
		    DMA_SetMajorLoopCount(eDMAChannel,Source->height);
		}

		DMA_MdestOffsetBit(eDMAChannel) = 1;
		DMA_MsrcOffsetBit(eDMAChannel) = 0; 	

		/* VIOLATION TO [ MISRA 16.9 ] See 16.9-1*/
		Graphics_TCB[eDMAChannel] =  callback; 
		DMA_SetCallback(eDMAChannel, Graphics_TransformationCallback);
	    DMA_EnableRequest(eDMAChannel, (uint8_t)(DMA_REQ_ALWAYS_START + eDMAChannel));
	    DMA_SetReady(eDMAChannel);			

    }
    return error1;
}


/**
* \brief	Graphics_Rotate90MirrorV - Combined operation of Rotate90 and MirrorV
* \author	IM, b06623
* \param	uint8_t Channel, it is the DMA channel to be used.
* \param	const Graphics_Object_t* Source: Is the graphic object Source to be copied
* \param	const Graphics_Object_t* target: Is the graphic object Destination
* \return	DMA_ErrorType, possible resutls: GRAPHICS_ERROR_OK, GRAPHICS_ERROR_FAIL
* \todo
*/
Graphics_ErrorType Graphics_Rotate90MirrorV( uint8_t  eDMAChannel, Graphics_Object_t*  Source, Graphics_Object_t*  Target, Graphics_CallbackType callback)
{
    DMA_ErrorType	error;
    Graphics_ErrorType 	error1;
    uint32_t		tmp;

    error1 = GRAPHICS_ERROR_OK;
    error  = DMA_InitChannel( eDMAChannel );
    /* Calculate pixel size */
    tmp = 8u;
    Graphics_Pixel2Bytes(Target->BPP, &tmp);
    
    if( error != DMA_ERROR_OK )
    {
		error1 = GRAPHICS_ERROR_FAIL;
    }
    else if(Source->width > Target->height)
    {
		DMA_UnInitChannel(eDMAChannel);
		error1 = GRAPHICS_ERROR_FAIL;
    }
    else if((tmp < 8u) || (tmp == 24u))
    {
		DMA_UnInitChannel(eDMAChannel);
		error1 = GRAPHICS_ERROR_FAIL;
    }
    else
    {
		if(tmp == 8u)
		{
		    DMA_DestinationSize(eDMAChannel) = 0u;
		    DMA_SourceSize(eDMAChannel) = 0u;
		    tmp = 1u;
		}
		else if(tmp == 16u)
		{
		    DMA_DestinationSize(eDMAChannel) = 1u;
		    DMA_SourceSize(eDMAChannel) = 1u;
		    tmp = 2u;
		}
		else /* if(tmp == 32) */
		{
		   DMA_DestinationSize(eDMAChannel) = 2u;
		   DMA_SourceSize(eDMAChannel) = 2u;
		   tmp = 4u;
		}
	
		DMA_SourceOffset(eDMAChannel) = (int16_t)tmp; 
		DMA_DestinationOffset(eDMAChannel) = (int16_t)((int16_t)Target->width*(int16_t)tmp);	
		
		DMA_MinorLoopCount(eDMAChannel) = ( uint32_t )Source->width*tmp;
		DMA_MloopOffsetValue(eDMAChannel) = (int32_t)tmp - ((int32_t)Source->width * (int32_t)Target->width * (int32_t)tmp);
		
		DMA_SourceAddress(eDMAChannel) = Source->address; 
		DMA_DestinationAddress(eDMAChannel) = Target->address;
		
		if( Target->width < Source->height )
		{
		    DMA_SetMajorLoopCount(eDMAChannel,Target->width);
		}
		else
		{
		    DMA_SetMajorLoopCount(eDMAChannel,Source->height);
		}

		DMA_MdestOffsetBit(eDMAChannel) = 1;
		DMA_MsrcOffsetBit(eDMAChannel) = 0; 	

		/* VIOLATION TO [ MISRA 16.9 ] See 16.9-1*/
		Graphics_TCB[eDMAChannel] =  callback; 
		DMA_SetCallback(eDMAChannel, Graphics_TransformationCallback);
	    DMA_EnableRequest(eDMAChannel, (uint8_t)(DMA_REQ_ALWAYS_START + eDMAChannel));
	    DMA_SetReady(eDMAChannel);
    }
    return error1;
}

/**
* \brief	Graphics_Rotate90MirrorH - Combined operation of Rotate90 and MirrorH
* \author	IM, b06623
* \param	uint8_t Channel, it is the DMA channel to be used.
* \param	const Graphics_Object_t* Source: Is the graphic object Source to be copied
* \param	const Graphics_Object_t* target: Is the graphic object Destination
* \return	DMA_ErrorType, possible resutls: GRAPHICS_ERROR_OK, GRAPHICS_ERROR_FAIL
* \todo
*/
Graphics_ErrorType Graphics_Rotate90MirrorH( uint8_t  eDMAChannel, Graphics_Object_t*  Source, Graphics_Object_t*  Target, Graphics_CallbackType callback)
{
    DMA_ErrorType	error;
    Graphics_ErrorType 	error1;
    uint32_t		tmp;

    error1 = GRAPHICS_ERROR_OK;
    error  = DMA_InitChannel( eDMAChannel );
    /* Calculate pixel size */
    tmp = 8u;
    Graphics_Pixel2Bytes(Target->BPP, &tmp);
    
    if( error != DMA_ERROR_OK )
    {
		error1 = GRAPHICS_ERROR_FAIL;
    }
    else if(Source->width > Target->height)
    {
		DMA_UnInitChannel(eDMAChannel);
		error1 = GRAPHICS_ERROR_FAIL;
    }
    else if((tmp < 8u) || (tmp == 24u))
    {
		DMA_UnInitChannel(eDMAChannel);
		error1 = GRAPHICS_ERROR_FAIL;
    }
    else
    {
		if(tmp == 8u)
		{
		    DMA_DestinationSize(eDMAChannel) = 0u;
		    DMA_SourceSize(eDMAChannel) = 0u;
		    tmp = 1u;
		}
		else if(tmp == 16u)
		{
		    DMA_DestinationSize(eDMAChannel) = 1u;
		    DMA_SourceSize(eDMAChannel) = 1u;
		    tmp = 2u;
		}
		else /* if(tmp == 32) */
		{
		   DMA_DestinationSize(eDMAChannel) = 2u;
		   DMA_SourceSize(eDMAChannel) = 2u;
		   tmp = 4u;
		} 
	
		DMA_SourceOffset(eDMAChannel) = (int16_t)tmp; 
		DMA_DestinationOffset(eDMAChannel) = (int16_t)(-1*(int16_t)Target->width*(int16_t)tmp);	
		
		DMA_MinorLoopCount(eDMAChannel) = (uint32_t)Source->width*tmp;
		DMA_MloopOffsetValue(eDMAChannel) = ((int32_t)Source->width * (int32_t)Target->width * (int32_t)tmp) - (int32_t)tmp;
		
		DMA_SourceAddress(eDMAChannel) = Source->address; 
		DMA_DestinationAddress(eDMAChannel) = Target->address + (uint32_t)DMA_MloopOffsetValue(eDMAChannel);
		
		if( Target->width < Source->height )
		{
		    DMA_SetMajorLoopCount(eDMAChannel,Target->width);
		}
		else
		{
		    DMA_SetMajorLoopCount(eDMAChannel,Source->height);
		}

		DMA_MdestOffsetBit(eDMAChannel) = 1;
		DMA_MsrcOffsetBit(eDMAChannel) = 0; 	

		/* VIOLATION TO [ MISRA 16.9 ] See 16.9-1*/
		Graphics_TCB[eDMAChannel] =  callback; 
		DMA_SetCallback(eDMAChannel, Graphics_TransformationCallback);
	    DMA_EnableRequest(eDMAChannel, (uint8_t)(DMA_REQ_ALWAYS_START + eDMAChannel));
	    DMA_SetReady(eDMAChannel);
    }
    return error1;
}

/**
* \brief	Graphics_Decode8bits - Decodes with the CPU a RLE coded stream
* \author	IM, b06623
* \param	uint8_t *buf: Output buffer where decoded info will be written
* \param	const uint8_t *compressed: Source of RLE coded information
* \param	int16_t len: size of the information to be decoded
* \return	void
* \todo
*/
void Graphics_Decode8bits(uint8_t *buf, const uint8_t *compressed, int32_t len)
{
    uint8_t cmd;
    uint16_t seqlen;
    int16_t i;

    while(len > 0)
    {
		cmd = *compressed;
	    *compressed++; 
		seqlen = (uint16_t)((cmd & 0x7fu) + 1u);


		if(cmd & 0x80u)
		{
		    /* rle encoded sequence */
		    for(i=0u; i<seqlen; i++)
		    {
				*buf = *compressed;
	        	*buf++; 
		    }
		    compressed++;
		}
		else
		{
		    /* raw sequence */
		    for(i=0u; i<seqlen; i++)
		    {
				*buf = *compressed;
	        	*buf++; 
	        	*compressed++; 
		    }
		}
		len -= (int32_t)seqlen;
    }
}

/**
* \brief	Graphics_Decode8bits - Decodes with the CPU and the DMA a RLE coded stream
* \author	IM, b06623
* \param	uint8_t Channel, it is the DMA channel to be used.
* \param	uint8_t *buf: Output buffer where decoded info will be written
* \param	const uint8_t *compressed: Source of RLE coded information
* \param	int16_t len: size of the information to be decoded
* \return	void
* \todo
*/
void Graphics_Decode8bitsDMA(uint8_t eDMAChannel, uint8_t *buf, const uint8_t *compressed, int32_t len)
{
    uint8_t cmd;
    uint16_t seqlen;
    DMA_ErrorType	error;

    error = DMA_InitChannel(eDMAChannel);
    
    if( error != DMA_ERROR_OK )
	{
		return;
	}
    
	DMA_SourceAddress(eDMAChannel) = (uint32_t)compressed; 
	DMA_DestinationAddress(eDMAChannel) = (uint32_t)buf;
	DMA_SourceOffset(eDMAChannel) = 1;
	DMA_DestinationOffset(eDMAChannel) = 1;	
	DMA_SetMajorLoopCount(eDMAChannel,1);

    while(len > 0)
    {
		cmd = *( ( uint8_t * )( DMA_SourceAddress(eDMAChannel) ) );
		DMA_SourceAddress( eDMAChannel )++;
		seqlen = (uint16_t)((cmd & 0x7fu) + 1u);

		if(cmd & 0x80u)
		{
		    DMA_LastSourceAdjustment( eDMAChannel ) = 1u;
	        /* Convertion to a wider integer */ 
		    DMA_MinorLoopCount( eDMAChannel ) = seqlen;
		    DMA_SourceOffset( eDMAChannel ) = 0u;

		    DMA_SWStart( eDMAChannel );
		    while( DMA_Done( eDMAChannel ) == 0u )
		    {
				;
		    }
		}
		else
		{
	    	DMA_LastSourceAdjustment( eDMAChannel ) = 0u;
	        /* Convertion to a wider integer */  
		    DMA_MinorLoopCount( eDMAChannel ) = seqlen;
		    DMA_SourceOffset( eDMAChannel ) = 1u;

		    DMA_SWStart( eDMAChannel );
		    while( DMA_Done( eDMAChannel ) == 0u )
		    {
				;
		    }
		}

		len -= (int32_t)seqlen;
    }

    DMA_UnInitChannel( eDMAChannel ); 
}




/********************** Pure DMA RLE Decode Methods ********************************/
#ifndef RLE
/* proto */
uint32_t DMA_RLECounts[DMA_MAX_DMACHANNELS];

/* manages callbacks and channel uninitialization */
static void Graphics_RLECallback(uint8_t channel)
{
	uint8_t channelLink;		
	channelLink = (uint8_t)(0x7F&(EDMA.TCD[(channel)].BITER>>9));		
	if(DMA_RLECounts[channel]>0)
	{
		DMA_SetMajorLoopCountLink(channel, channelLink, 511);
		DMA_SetMajorLoopCountLink(channelLink, channel, 511);
		DMA_RLECounts[channel]--;
		DMA_SetCallback(channel, Graphics_RLECallback);
		DMA_SWStart(channel);
	}
	else
	{	
		DMA_UnInitChannel(channelLink);
		DMA_UnInitChannel(channel);	
		if(Graphics_TCB[channel] != NULL_PTR)
		{
			Graphics_TCB[channel]();	
		}
	}
}


void Graphics_RLEDMA
(
	uint8_t eDMAChannel,
	uint8_t eDMAChannel0, 
	Graphics_Object_t* src,
	Graphics_Object_t* dest,
	Graphics_CallbackType callback
)
{
    DMA_ErrorType	error;
    uint32_t size;
    const uint8_t *cmd;
    const uint8_t *compressed;
    uint8_t *buf;

    error = DMA_InitChannel(eDMAChannel);
    
    if( error != DMA_ERROR_OK )
	{
		return;
	}
	
	error = DMA_InitChannel(eDMAChannel0);
	
	if( error != DMA_ERROR_OK )
	{
		DMA_UnInitChannel(eDMAChannel);
		return;
	}
    
    cmd = (uint8_t *)dest->address;    
    buf = (uint8_t *)src->address;
    
    /* Init Control DMA channel */
    size = (cmd[0]<<24) | (cmd[1]<<16) | (cmd[2]<<8) | (cmd[3]);
    compressed = (uint8_t *)(&cmd[(size*2)+4]);
    DMA_RLECounts[eDMAChannel] = size/510;
    size = 1 + size - (510*DMA_RLECounts[eDMAChannel]);
    
    DMA_SourceAddress(eDMAChannel) = ((uint32_t)cmd) + 4; 
	DMA_DestinationAddress(eDMAChannel) = (uint32_t)(&DMA_SourceOffset(eDMAChannel0)) + 1;
	DMA_SourceOffset(eDMAChannel) = 1;
	DMA_DestinationOffset(eDMAChannel) = 4;
	DMA_MinorLoopCount( eDMAChannel ) = 2;	
	DMA_MdestOffsetBit(eDMAChannel) = 1;
	DMA_MsrcOffsetBit(eDMAChannel) = 0; 
	DMA_MloopOffsetValue(eDMAChannel) = -8;	
	DMA_SetMajorLoopCountLink(eDMAChannel, eDMAChannel0, (uint16_t)size);
	DMA_LastDestinationAdjustment(eDMAChannel) = -8; 
	DMA_LastSourceAdjustment(eDMAChannel) = -2;	
	/* Init burst DMA channel */
    DMA_SourceAddress(eDMAChannel0) = ((uint32_t)compressed); 
	DMA_DestinationAddress(eDMAChannel0) = (uint32_t)buf;
	DMA_DestinationOffset(eDMAChannel0) = 1;		
	DMA_MdestOffsetBit(eDMAChannel0) = 0;
	DMA_MsrcOffsetBit(eDMAChannel0) = 1; 
	DMA_MloopOffsetValue(eDMAChannel0) = 1;
	DMA_SetMajorLoopCountLink(eDMAChannel0, eDMAChannel, (uint16_t)size);
	// VIOLATION TO [ MISRA 16.9 ] See 16.9-1
	Graphics_TCB[eDMAChannel] =  callback; 
	DMA_SetCallback(eDMAChannel, Graphics_RLECallback);
    DMA_SetReady(eDMAChannel);
}
#endif

#if 0
///***** Confidential IP, scale method using pure DMA
extern const uint8_t TEST_Bitmap0[];
extern const unsigned short mult_table[];
extern const unsigned short sum_table[];
uint8_t offset_fill[4] = {12,4,0,0};
uint8_t Graphics_mult[400];
uint8_t Graphics_offset[800];
uint32_t Graphics_address[100];
uint32_t Graphics_reload[2];
uint8_t Graphics_out[100];
void Graphics_Scale
(
	uint8_t factor,
	uint16_t offset,
	Graphics_Object_t* src,
	Graphics_Object_t* dst,
	DMA_CallbackType callback
)
{
	int i;
	int j;
	int k;
	uint8_t tmp0;
	uint8_t tmp1;
	uint8_t tmp2;
	//fill mult and offset vectors:
	uint16_t width;
	uint16_t height;	
	
	tmp0 = factor;
	tmp1 = 0;
	tmp2 = 0;
	
	width = 100;
	height = 100;	
	
	for(i =0; i<400; i++)
		Graphics_offset[1 + i*2] = offset_fill[i%3];
	
	Graphics_address[0] = 0x60000000;		
	for(i =1; i<100; i++)
		Graphics_address[i] = Graphics_address[i-1] + width;	
	
	Graphics_reload[0] = (uint32_t)Graphics_mult;
	Graphics_reload[1] = (uint32_t)Graphics_offset;
	
	
	j = 0;
	i = 0;
	k = 0;


	while(i<width)
	{		
		if(tmp1 == 0)
		{
			//check factors, must be always 3
			if(tmp2 == 2)
			{
				Graphics_mult[j] = 0;
				Graphics_offset[k] = 0;
				j++;
				k+=2;	
			}
			tmp2 = 0;		
			Graphics_mult[j] = tmp0;
		}							
		else 
			if(tmp1<tmp0)		
				Graphics_mult[j] = tmp1;
			else
				Graphics_mult[j] = tmp0;
		
		// check unity
		tmp1 -= Graphics_mult[j]; 
		// check factor
		tmp0 -= Graphics_mult[j];
		if(tmp0==0)
		{
			tmp0 = factor;
			i++;
			Graphics_offset[k] = 1;						
		}
		else
		{
			Graphics_offset[k] = 0;
		}					
		j++;
		k+=2;
		
		tmp2++;	//count factors							
	}
	//tables has been constructed
	//configure DMAs

	DMA_UnInitChannel(0);
	DMA_UnInitChannel(1);
	DMA_UnInitChannel(2);
	DMA_UnInitChannel(3);
	DMA_UnInitChannel(4);
	DMA_UnInitChannel(5);
	DMA_UnInitChannel(6);
		
	DMA_InitChannel(0);
	DMA_InitChannel(1);
	DMA_InitChannel(2);
	DMA_InitChannel(3);
	DMA_InitChannel(4);
	DMA_InitChannel(5);
	DMA_InitChannel(6);
	
	j = j + 3 - (j%3);
	
	//DMA which triggers lines operations (5)
	DMA_SourceAddress(5) = (uint32_t)Graphics_address; //the table with dest line start
	DMA_DestinationAddress(5) = (uint32_t)(&DMA_DestinationAddress(4)); //update destination address 	
	DMA_SourceOffset(5) = 4;	//go over each element of the vector
	DMA_DestinationOffset(5) = 0; // always same destination	
	DMA_MinorLoopCount(5) = 4;	// trigger the other channel after 1 word transfer
	DMA_SetMajorLoopCountLink(5, 0, (uint16_t)height); // number of lines from the source object	
	DMA_LastDestinationAdjustment(5) = 0; 
	DMA_LastSourceAdjustment(5) = 0;
	DMA_DestinationSize(5) = 2; //32bit
	DMA_SourceSize(5) = 2; //32bit	


	//DMA which transfers the factor table (0)
	
	//DMA_SourceAddress(0) = (uint32_t)Graphics_mult; //the table with the byte factors 
	DMA_SourceAddress(0) = (uint32_t)Graphics_mult; //the table with the byte factors 
	DMA_DestinationAddress(0) = 3 + (uint32_t)(&DMA_SourceAddress(3)); //lowest byte of the address of the mult dma 	
	DMA_SourceOffset(0) = 1;	//go factor by factor; 3 per multiply+add
	DMA_DestinationOffset(0) = 0; // always same destination	
	DMA_MinorLoopCount(0) = 1;	// trigger the other channel after 1 byte transfer
	DMA_SetMajorLoopCountLink(0, 1, j+1); // number of pixels, plus 1 dummy which restarts itself
	DMA_LastDestinationAdjustment(0) = 0; 
	DMA_LastSourceAdjustment(0) = -j-1; //restore source address
	EDMA.TCD[0].MAJORE_LINK = 1;
	EDMA.TCD[0].MAJORLINKCH = 5; //restart dma 5

	
	//DMA which transfers the control offset to the DMA which transfers the image pixels
	//and the multiplier destination DMA for correct function with the summation dma
	DMA_SourceAddress(1) = (uint32_t)Graphics_offset; //the table with the byte factors 
	DMA_DestinationAddress(1) = 1 + (uint32_t)(&DMA_SourceOffset(2)); //offset register of the DMA transfer pixels
	DMA_SourceOffset(1) = 1;	//go factor by factor; 3 per multiply+add
	DMA_DestinationOffset(1) = 48; // go to the other DMA
	DMA_MinorLoopCount(1) = 2;	// transfer two bytes
	
	DMA_MloopOffsetValue(1) = -96; //go back
	DMA_MdestOffsetBit(1) = 1;
	DMA_MsrcOffsetBit(1) = 0; 	
		
	DMA_SetMajorLoopCountLink(1, 2, j); // no minor loop link needed, just 1 major loop.
	DMA_LastDestinationAdjustment(1) = -96; //go back to the original address
	DMA_LastSourceAdjustment(1) = -j*2;
	EDMA.TCD[1].MAJORE_LINK = 1;
	EDMA.TCD[1].MAJORLINKCH = 2;
	
	//DMA which transfers the Pixels to the multiplier
	DMA_SourceAddress(2) =  (uint32_t)TEST_Bitmap0; //the table with the byte factors 
	DMA_DestinationAddress(2) = 2 + (uint32_t)(&DMA_SourceAddress(3)); //3rd byte of the address of the mult dma 	
	DMA_SourceOffset(2) = 0;	//this will be controlled by the other DMA	
	DMA_DestinationOffset(2) = 0; // always same destination	
	DMA_MinorLoopCount(2) = 1;	// trigger the other channel after 1 byte transfer
	DMA_SetMajorLoopCountLink(2, 3, j); // the number of lines, trigger channel 2, DMA pixels	
	DMA_LastDestinationAdjustment(2) = 0; 
	DMA_LastSourceAdjustment(2) = 0; //potentially can grab a subimage by changing this
	EDMA.TCD[2].MAJORE_LINK = 1;
	EDMA.TCD[2].MAJORLINKCH = 3;
	
	//DMA which multiplies pixels and factors, it sends them to the sum DMA
	DMA_SourceAddress(3) = (uint32_t)mult_table; //the table with the byte factors 
	DMA_DestinationAddress(3) = (uint32_t)(&DMA_SourceOffset(4)); //offset register of the DMA transfer pixels	
	DMA_SourceOffset(3) = 0;	//no advance
	DMA_DestinationOffset(3) = 0; //this will be controlled by the other DMA				
	DMA_MinorLoopCount(3) = 2;	// trigger the other channel after 1 16bit transfer
		
	DMA_SetMajorLoopCount(3, 1); // no minor loop link needed, just 1 major loop.
	DMA_LastDestinationAdjustment(3) = 0; 
	DMA_LastSourceAdjustment(3) = 0;	
	DMA_DestinationSize(3) = 1; //16bit
	DMA_SourceSize(3) = 1; //16bit
	EDMA.TCD [3].DMOD = 4;
	EDMA.TCD[3].MAJORE_LINK = 1;
	EDMA.TCD[3].MAJORLINKCH = 4;
	

	//DMA which sums the multiply result and paste it on the final vector
	DMA_SourceAddress(4) = (uint32_t)sum_table;  //the table with the byte results
	DMA_DestinationAddress(4) = 0x60000000; //test test test		
	DMA_SourceOffset(4) = 0;	//no advance
	DMA_DestinationOffset(4) = 0; // no advance	
	DMA_MinorLoopCount(4) = 1;	// sum each time something comes up
	
	//DMA_MloopOffsetValue(4) = 0xffff; //go back
	//DMA_MdestOffsetBit(4) = 0;
	//DMA_MsrcOffsetBit(4) = 1; 	
	
	DMA_LastDestinationAdjustment(4) = 1;	// advance 1 byte after the three sums 
	DMA_LastSourceAdjustment(4) = 0;		// stay
	
	//DMA_SetMajorLoopCountLink(4, 0, 3); 	
	EDMA.TCD[4].BITERE_LINK = 1;	
	EDMA.TCD[4].CITERE_LINK = 1;	
	EDMA.TCD[4].BITER = ((0<<9)|(3));	
    EDMA.TCD[4].CITER = ((0<<9)|(1));	
	
	EDMA.TCD[4].MAJORE_LINK = 1;
	EDMA.TCD[4].MAJORLINKCH = 0;
	EDMA.TCD[4].SMOD = 16;
	
	DMA_BWC(4) = 2;
	
	DMA_SetCallback(5, callback);
	
	DMA_SWStart(5); 			
	
	
}

#endif


/* returns a factor based on the bpp value */
static void Graphics_Pixel2Bytes(uint8_t BPP, uint32_t *size)
{
    switch(BPP)
    {
	case GRAPHICS_1BPP:
	    *size = *size/8u;
	    break;
	case GRAPHICS_2BPP:
	    *size = *size/4u;
	    break;
	case GRAPHICS_4BPP:
	case GRAPHICS_T4BPP:
	case GRAPHICS_L4BPP:
	    *size = *size/2u;
	    break;
	case GRAPHICS_565RGB:
	case GRAPHICS_1555ARGB:
	case GRAPHICS_4444ARGB:
    #ifdef DCU3	
        case GRAPHICS_YCBCR422:
        case GRAPHICS_APAL8:
    #endif    
    #ifdef DCU4	
        case GRAPHICS_YCBCR422:
        case GRAPHICS_APAL8:
    #endif    
	    *size = *size*2u;
	    break;
	case GRAPHICS_24BPP:
	    *size = *size*3u;
	    break;
	case GRAPHICS_32BPP:
	    *size = *size*4u;
	    break;
	case GRAPHICS_8BPP:
	case GRAPHICS_T8BPP:
	case GRAPHICS_L8BPP:	    
	default:
	    /* Do nothing */
	    break;
    }
}

/* manages callbacks and channel uninitialization */
static void Graphics_TransformationCallback(uint8_t channel)
{
	DMA_UnInitChannel(channel);	
	if(Graphics_TCB[channel] != NULL_PTR)
	{
		Graphics_TCB[channel]();	
	}
}